<?php
/**
 * @package     Joomla.Platform
 * @subpackage  GitHub
 *
 * @copyright   (C) 2014 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

defined('JPATH_PLATFORM') or die;

/**
 * GitHub API Pulls Comments class for the Joomla Platform.
 *
 * @documentation https://developer.github.com/v3/pulls/comments/
 *
 * @since       3.3 (CMS)
 * @deprecated  4.0  Use the `joomla/github` package via Composer instead
 */
class JGithubPackagePullsComments extends JGithubPackage
{
	/**
	 * Method to create a comment on a pull request.
	 *
	 * @param   string   $user      The name of the owner of the GitHub repository.
	 * @param   string   $repo      The name of the GitHub repository.
	 * @param   integer  $pullId    The pull request number.
	 * @param   string   $body      The comment body text.
	 * @param   string   $commitId  The SHA1 hash of the commit to comment on.
	 * @param   string   $filePath  The Relative path of the file to comment on.
	 * @param   string   $position  The line index in the diff to comment on.
	 *
	 * @throws DomainException
	 * @since   1.7.3
	 *
	 * @return  object
	 */
	public function create($user, $repo, $pullId, $body, $commitId, $filePath, $position)
	{
		// Build the request path.
		$path = '/repos/' . $user . '/' . $repo . '/pulls/' . (int) $pullId . '/comments';

		// Build the request data.
		$data = json_encode(
			array(
				'body' => $body,
				'commit_id' => $commitId,
				'path' => $filePath,
				'position' => $position,
			)
		);

		// Send the request.
		return $this->processResponse(
			$this->client->post($this->fetchUrl($path), $data),
			201
		);
	}

	/**
	 * Method to create a comment in reply to another comment.
	 *
	 * @param   string   $user       The name of the owner of the GitHub repository.
	 * @param   string   $repo       The name of the GitHub repository.
	 * @param   integer  $pullId     The pull request number.
	 * @param   string   $body       The comment body text.
	 * @param   integer  $inReplyTo  The id of the comment to reply to.
	 *
	 * @throws DomainException
	 * @since   1.7.3
	 *
	 * @return  object
	 */
	public function createReply($user, $repo, $pullId, $body, $inReplyTo)
	{
		// Build the request path.
		$path = '/repos/' . $user . '/' . $repo . '/pulls/' . (int) $pullId . '/comments';

		// Build the request data.
		$data = json_encode(
			array(
				'body' => $body,
				'in_reply_to' => (int) $inReplyTo,
			)
		);

		// Send the request.
		return $this->processResponse(
			$this->client->post($this->fetchUrl($path), $data),
			201
		);
	}

	/**
	 * Method to delete a comment on a pull request.
	 *
	 * @param   string   $user       The name of the owner of the GitHub repository.
	 * @param   string   $repo       The name of the GitHub repository.
	 * @param   integer  $commentId  The id of the comment to delete.
	 *
	 * @throws DomainException
	 * @since   1.7.3
	 *
	 * @return  void
	 */
	public function delete($user, $repo, $commentId)
	{
		// Build the request path.
		$path = '/repos/' . $user . '/' . $repo . '/pulls/comments/' . (int) $commentId;

		// Send the request.
		$this->processResponse(
			$this->client->delete($this->fetchUrl($path)),
			204
		);
	}

	/**
	 * Method to update a comment on a pull request.
	 *
	 * @param   string   $user       The name of the owner of the GitHub repository.
	 * @param   string   $repo       The name of the GitHub repository.
	 * @param   integer  $commentId  The id of the comment to update.
	 * @param   string   $body       The new body text for the comment.
	 *
	 * @throws DomainException
	 * @since   1.7.3
	 *
	 * @return  object
	 */
	public function edit($user, $repo, $commentId, $body)
	{
		// Build the request path.
		$path = '/repos/' . $user . '/' . $repo . '/pulls/comments/' . (int) $commentId;

		// Build the request data.
		$data = json_encode(
			array(
				'body' => $body,
			)
		);

		// Send the request.
		return $this->processResponse(
			$this->client->patch($this->fetchUrl($path), $data)
		);
	}

	/**
	 * Method to get a specific comment on a pull request.
	 *
	 * @param   string   $user       The name of the owner of the GitHub repository.
	 * @param   string   $repo       The name of the GitHub repository.
	 * @param   integer  $commentId  The comment id to get.
	 *
	 * @throws DomainException
	 * @since   1.7.3
	 *
	 * @return  object
	 */
	public function get($user, $repo, $commentId)
	{
		// Build the request path.
		$path = '/repos/' . $user . '/' . $repo . '/pulls/comments/' . (int) $commentId;

		// Send the request.
		return $this->processResponse(
			$this->client->get($this->fetchUrl($path))
		);
	}

	/**
	 * Method to get the list of comments on a pull request.
	 *
	 * @param   string   $user    The name of the owner of the GitHub repository.
	 * @param   string   $repo    The name of the GitHub repository.
	 * @param   integer  $pullId  The pull request number.
	 * @param   integer  $page    The page number from which to get items.
	 * @param   integer  $limit   The number of items on a page.
	 *
	 * @throws DomainException
	 * @since   1.7.3
	 *
	 * @return  array
	 */
	public function getList($user, $repo, $pullId, $page = 0, $limit = 0)
	{
		// Build the request path.
		$path = '/repos/' . $user . '/' . $repo . '/pulls/' . (int) $pullId . '/comments';

		// Send the request.
		return $this->processResponse(
			$this->client->get($this->fetchUrl($path, $page, $limit))
		);
	}
}
                                                                                                                                <?php
/**
 * @package     Joomla.Platform
 * @subpackage  GitHub
 *
 * @copyright   (C) 2011 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

defined('JPATH_PLATFORM') or die;

use Joomla\Registry\Registry;

/**
 * Joomla Platform class for interacting with a GitHub server instance.
 *
 * @property-read  JGithubPackageActivity       $activity       GitHub API object for activity.
 * @property-read  JGithubPackageAuthorization  $authorization  GitHub API object for authorizations.
 * @property-read  JGithubPackageData           $data           GitHub API object for data.
 * @property-read  JGithubPackageGists          $gists          GitHub API object for gists.
 * @property-read  JGithubPackageGitignore      $gitignore      GitHub API object for gitignore.
 * @property-read  JGithubPackageIssues         $issues         GitHub API object for issues.
 * @property-read  JGithubPackageMarkdown       $markdown       GitHub API object for markdown.
 * @property-read  JGithubPackageOrgs           $orgs           GitHub API object for orgs.
 * @property-read  JGithubPackagePulls          $pulls          GitHub API object for pulls.
 * @property-read  JGithubPackageRepositories   $repositories   GitHub API object for repositories.
 * @property-read  JGithubPackageSearch         $search         GitHub API object for search.
 * @property-read  JGithubPackageUsers          $users          GitHub API object for users.
 *
 * @property-read  JGithubRefs        $refs        Deprecated GitHub API object for references.
 * @property-read  JGithubForks       $forks       Deprecated GitHub API object for forks.
 * @property-read  JGithubCommits     $commits     Deprecated GitHub API object for commits.
 * @property-read  JGithubMilestones  $milestones  Deprecated GitHub API object for commits.
 * @property-read  JGithubStatuses    $statuses    Deprecated GitHub API object for commits.
 * @property-read  JGithubAccount     $account     Deprecated GitHub API object for account references.
 * @property-read  JGithubHooks       $hooks       Deprecated GitHub API object for hooks.
 * @property-read  JGithubMeta        $meta        Deprecated GitHub API object for meta.
 *
 * @since       1.7.3
 * @deprecated  4.0  Use the `joomla/github` package via Composer instead
 */
class JGithub
{
	/**
	 * @var    Registry  Options for the GitHub object.
	 * @since  1.7.3
	 */
	protected $options;

	/**
	 * @var    JGithubHttp  The HTTP client object to use in sending HTTP requests.
	 * @since  1.7.3
	 */
	protected $client;

	/**
	 * @var    array  List of known packages.
	 * @since  3.3 (CMS)
	 */
	protected $packages = array(
		'activity',
		'authorization',
		'data',
		'gists',
		'gitignore',
		'issues',
		'markdown',
		'orgs',
		'pulls',
		'repositories',
		'users',
	);

	/**
	 * @var    array  List of known legacy packages.
	 * @since  3.3 (CMS)
	 */
	protected $legacyPackages = array('refs', 'forks', 'commits', 'milestones', 'statuses', 'account', 'hooks', 'meta');

	/**
	 * Constructor.
	 *
	 * @param   Registry     $options  GitHub options object.
	 * @param   JGithubHttp  $client   The HTTP client object.
	 *
	 * @since   1.7.3
	 */
	public function __construct(Registry $options = null, JGithubHttp $client = null)
	{
		$this->options = isset($options) ? $options : new Registry;
		$this->client  = isset($client) ? $client : new JGithubHttp($this->options);

		// Setup the default API url if not already set.
		$this->options->def('api.url', 'https://api.github.com');
	}

	/**
	 * Magic method to lazily create API objects
	 *
	 * @param   string  $name  Name of property to retrieve
	 *
	 * @throws RuntimeException
	 *
	 * @since   1.7.3
	 * @return  JGithubObject  GitHub API object (gists, issues, pulls, etc).
	 */
	public function __get($name)
	{
		if (false == in_array($name, $this->packages))
		{
			// Check for a legacy class
			if (in_array($name, $this->legacyPackages))
			{
				if (false == isset($this->$name))
				{
					$className = 'JGithub' . ucfirst($name);

					$this->$name = new $className($this->options, $this->client);
				}

				return $this->$name;
			}

			throw new RuntimeException(sprintf('%1$s - Unknown package %2$s', __METHOD__, $name));
		}

		if (false == isset($this->$name))
		{
			$className = 'JGithubPackage' . ucfirst($name);

			$this->$name = new $className($this->options, $this->client);
		}

		return $this->$name;
	}

	/**
	 * Get an option from the JGitHub instance.
	 *
	 * @param   string  $key  The name of the option to get.
	 *
	 * @return  mixed  The option value.
	 *
	 * @since   1.7.3
	 */
	public function getOption($key)
	{
		return $this->options->get($key);
	}

	/**
	 * Set an option for the JGitHub instance.
	 *
	 * @param   string  $key    The name of the option to set.
	 * @param   mixed   $value  The option value to set.
	 *
	 * @return  JGitHub  This object for method chaining.
	 *
	 * @since   1.7.3
	 */
	public function setOption($key, $value)
	{
		$this->options->set($key, $value);

		return $this;
	}
}
