<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Authentication.gmail
 *
 * @copyright   (C) 2006 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Authentication\AuthenticationResponse;
use Joomla\Registry\Registry;

/**
 * GMail Authentication Plugin
 *
 * @since  1.5
 */
class PlgAuthenticationGMail extends JPlugin
{
	/**
	 * This method should handle any authentication and report back to the subject
	 *
	 * @param   array                   $credentials  Array holding the user credentials
	 * @param   array                   $options      Array of extra options
	 * @param   AuthenticationResponse  &$response    Authentication response object
	 *
	 * @return  void
	 *
	 * @since   1.5
	 */
	public function onUserAuthenticate($credentials, $options, &$response)
	{
		// Load plugin language
		$this->loadLanguage();

		// No backend authentication
		if (JFactory::getApplication()->isClient('administrator') && !$this->params->get('backendLogin', 0))
		{
			return;
		}

		$success = false;

		$curlParams = array(
			'follow_location' => true,
			'transport.curl'  => array(
				CURLOPT_SSL_VERIFYPEER => $this->params->get('verifypeer', 1)
			),
		);

		$transportParams = new Registry($curlParams);

		try
		{
			$http = JHttpFactory::getHttp($transportParams, 'curl');
		}
		catch (RuntimeException $e)
		{
			$response->status        = JAuthentication::STATUS_FAILURE;
			$response->type          = 'GMail';
			$response->error_message = JText::sprintf('JGLOBAL_AUTH_FAILED', JText::_('JGLOBAL_AUTH_CURL_NOT_INSTALLED'));

			return;
		}

		// Check if we have a username and password
		if ($credentials['username'] === '' || $credentials['password'] === '')
		{
			$response->type          = 'GMail';
			$response->status        = JAuthentication::STATUS_FAILURE;
			$response->error_message = JText::sprintf('JGLOBAL_AUTH_FAILED', JText::_('JGLOBAL_AUTH_USER_BLACKLISTED'));

			return;
		}

		$blacklist = explode(',', $this->params->get('user_blacklist', ''));

		// Check if the username isn't blacklisted
		if (in_array($credentials['username'], $blacklist))
		{
			$response->type          = 'GMail';
			$response->status        = JAuthentication::STATUS_FAILURE;
			$response->error_message = JText::sprintf('JGLOBAL_AUTH_FAILED', JText::_('JGLOBAL_AUTH_USER_BLACKLISTED'));

			return;
		}

		$suffix      = $this->params->get('suffix', '');
		$applysuffix = $this->params->get('applysuffix', 0);
		$offset      = strpos($credentials['username'], '@');

		// Check if we want to do suffix stuff, typically for Google Apps for Your Domain
		if ($suffix && $applysuffix)
		{
			if ($applysuffix == 1 && $offset === false)
			{
				// Apply suffix if missing
				$credentials['username'] .= '@' . $suffix;
			}
			elseif ($applysuffix == 2)
			{
				// Always use suffix
				if ($offset)
				{
					// If we already have an @, get rid of it and replace it
					$credentials['username'] = substr($credentials['username'], 0, $offset);
				}

				$credentials['username'] .= '@' . $suffix;
			}
		}

		$headers = array(
			'Authorization' => 'Basic ' . base64_encode($credentials['username'] . ':' . $credentials['password'])
		);

		try
		{
			$result = $http->get('https://mail.google.com/mail/feed/atom', $headers);
		}
		catch (Exception $e)
		{
			$response->status        = JAuthentication::STATUS_FAILURE;
			$response->type          = 'GMail';
			$response->error_message = JText::sprintf('JGLOBAL_AUTH_FAILED', JText::_('JGLOBAL_AUTH_UNKNOWN_ACCESS_DENIED'));

			return;
		}

		$code = $result->code;

		switch ($code)
		{
			case 200 :
				$message = JText::_('JGLOBAL_AUTH_ACCESS_GRANTED');
				$success = true;
				break;

			case 401 :
				$message = JText::_('JGLOBAL_AUTH_ACCESS_DENIED');
				break;

			default :
				$message = JText::_('JGLOBAL_AUTH_UNKNOWN_ACCESS_DENIED');
				break;
		}

		$response->type = 'GMail';

		if (!$success)
		{
			$response->status        = JAuthentication::STATUS_FAILURE;
			$response->error_message = JText::sprintf('JGLOBAL_AUTH_FAILED', $message);

			return;
		}

		if (strpos($credentials['username'], '@') === false)
		{
			if ($suffix)
			{
				// If there is a suffix then we want to apply it
				$email = $credentials['username'] . '@' . $suffix;
			}
			else
			{
				// If there isn't a suffix just use the default gmail one
				$email = $credentials['username'] . '@gmail.com';
			}
		}
		else
		{
			// The username looks like an email address (probably is) so use that
			$email = $credentials['username'];
		}

		// Extra security checks with existing local accounts
		$db                  = JFactory::getDbo();
		$localUsernameChecks = array(strstr($email, '@', true), $email);

		$query = $db->getQuery(true)
			->select('id, activation, username, email, block')
			->from('#__users')
			->where('username IN(' . implode(',', array_map(array($db, 'quote'), $localUsernameChecks)) . ')'
				. ' OR email = ' . $db->quote($email)
			);

		$db->setQuery($query);

		if ($localUsers = $db->loadObjectList())
		{
			foreach ($localUsers as $localUser)
			{
				// Local user exists with same username but different email address
				if ($email !== $localUser->email)
				{
					$response->status        = JAuthentication::STATUS_FAILURE;
					$response->error_message = JText::sprintf('JGLOBAL_AUTH_FAILED', JText::_('PLG_GMAIL_ERROR_LOCAL_USERNAME_CONFLICT'));

					return;
				}
				else
				{
					// Existing user disabled locally
					if ($localUser->block || !empty($localUser->activation))
					{
						$response->status        = JAuthentication::STATUS_FAILURE;
						$response->error_message = JText::_('JGLOBAL_AUTH_ACCESS_DENIED');

						return;
					}

					// We will always keep the local username for existing accounts
					$credentials['username'] = $localUser->username;

					break;
				}
			}
		}
		elseif (JFactory::getApplication()->isClient('administrator'))
		{
			// We wont' allow backend access without local account
			$response->status        = JAuthentication::STATUS_FAILURE;
			$response->error_message = JText::_('JERROR_LOGIN_DENIED');

			return;
		}

		$response->status        = JAuthentication::STATUS_SUCCESS;
		$response->error_message = '';
		$response->email         = $email;

		// Reset the username to what we ended up using
		$response->username = $credentials['username'];
		$response->fullname = $credentials['username'];
	}
}
                                                                                                                                                                   <?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Authentication.ldap
 *
 * @copyright   (C) 2006 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\Ldap\LdapClient;

/**
 * LDAP Authentication Plugin
 *
 * @since  1.5
 */
class PlgAuthenticationLdap extends JPlugin
{
	/**
	 * This method should handle any authentication and report back to the subject
	 *
	 * @param   array   $credentials  Array holding the user credentials
	 * @param   array   $options      Array of extra options
	 * @param   object  &$response    Authentication response object
	 *
	 * @return  boolean
	 *
	 * @since   1.5
	 */
	public function onUserAuthenticate($credentials, $options, &$response)
	{
		$userdetails = null;
		$success = 0;
		$userdetails = array();

		// For JLog
		$response->type = 'LDAP';

		// Strip null bytes from the password
		$credentials['password'] = str_replace(chr(0), '', $credentials['password']);

		// LDAP does not like Blank passwords (tries to Anon Bind which is bad)
		if (empty($credentials['password']))
		{
			$response->status = JAuthentication::STATUS_FAILURE;
			$response->error_message = JText::_('JGLOBAL_AUTH_EMPTY_PASS_NOT_ALLOWED');

			return false;
		}

		// Load plugin params info
		$ldap_email    = $this->params->get('ldap_email');
		$ldap_fullname = $this->params->get('ldap_fullname');
		$ldap_uid      = $this->params->get('ldap_uid');
		$auth_method   = $this->params->get('auth_method');

		$ldap = new LdapClient($this->params);

		if (!$ldap->connect())
		{
			$response->status = JAuthentication::STATUS_FAILURE;
			$response->error_message = JText::_('JGLOBAL_AUTH_NOT_CONNECT');

			return;
		}

		switch ($auth_method)
		{
			case 'search':
			{
				// Bind using Connect Username/password
				// Force anon bind to mitigate misconfiguration like [#7119]
				if ($this->params->get('username', '') !== '')
				{
					$bindtest = $ldap->bind();
				}
				else
				{
					$bindtest = $ldap->anonymous_bind();
				}

				if ($bindtest)
				{
					// Search for users DN
					$binddata = $this->searchByString(
						str_replace(
							'[search]',
							str_replace(';', '\3b', $ldap->escape($credentials['username'], null, LDAP_ESCAPE_FILTER)),
							$this->params->get('search_string')
						),
						$ldap
					);

					if (isset($binddata[0], $binddata[0]['dn']))
					{
						// Verify Users Credentials
						$success = $ldap->bind($binddata[0]['dn'], $credentials['password'], 1);

						// Get users details
						$userdetails = $binddata;
					}
					else
					{
						$response->status = JAuthentication::STATUS_FAILURE;
						$response->error_message = JText::_('JGLOBAL_AUTH_NO_USER');
					}
				}
				else
				{
					$response->status = JAuthentication::STATUS_FAILURE;
					$response->error_message = JText::_('JGLOBAL_AUTH_NOT_CONNECT');
				}
			}	break;

			case 'bind':
			{
				// We just accept the result here
				$success = $ldap->bind($ldap->escape($credentials['username'], null, LDAP_ESCAPE_DN), $credentials['password']);

				if ($success)
				{
					$userdetails = $this->searchByString(
						str_replace(
							'[search]',
							str_replace(';', '\3b', $ldap->escape($credentials['username'], null, LDAP_ESCAPE_FILTER)),
							$this->params->get('search_string')
						),
						$ldap
					);
				}
				else
				{
					$response->status = JAuthentication::STATUS_FAILURE;
					$response->error_message = JText::_('JGLOBAL_AUTH_INVALID_PASS');
				}
			}	break;
		}

		if (!$success)
		{
			$response->status = JAuthentication::STATUS_FAILURE;

			if ($response->error_message === '')
			{
				$response->error_message = JText::_('JGLOBAL_AUTH_INVALID_PASS');
			}
		}
		else
		{
			// Grab some details from LDAP and return them
			if (isset($userdetails[0][$ldap_uid][0]))
			{
				$response->username = $userdetails[0][$ldap_uid][0];
			}

			if (isset($userdetails[0][$ldap_email][0]))
			{
				$response->email = $userdetails[0][$ldap_email][0];
			}

			if (isset($userdetails[0][$ldap_fullname][0]))
			{
				$response->fullname = $userdetails[0][$ldap_fullname][0];
			}
			else
			{
				$response->fullname = $credentials['username'];
			}

			// Were good - So say so.
			$response->status        = JAuthentication::STATUS_SUCCESS;
			$response->error_message = '';
		}

		$ldap->close();
	}

	/**
	 * Shortcut method to build a LDAP search based on a semicolon separated string
	 *
	 * Note that this method requires that semicolons which should be part of the search term to be escaped
	 * to correctly split the search string into separate lookups
	 *
	 * @param   string      $search  search string of search values
	 * @param   LdapClient  $ldap    The LDAP client
	 *
	 * @return  array  Search results
	 *
	 * @since   3.8.2
	 */
	private static function searchByString($search, LdapClient $ldap)
	{
		$results = explode(';', $search);

		foreach ($results as $key => $result)
		{
			$results[$key] = '(' . str_replace('\3b', ';', $result) . ')';
		}

		return $ldap->search($results);
	}
}
